<?php

namespace App\Http\Controllers;

use App\Http\Requests\LegalProductsRequest;
use App\Models\LegalProducts;
use App\Models\LegalProductsStatus;
use App\Models\LegalProductsTypes;
use App\Models\LegalProductsCategories;
use Elasticsearch\ClientBuilder;
use Illuminate\Foundation\Console\StorageLinkCommand;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Inertia\Inertia;
use Spatie\QueryBuilder\QueryBuilder;
use Illuminate\Support\Facades\Storage;
use Spatie\Tags\Tag;

class LegalProductsController extends Controller
{
    /**
     * Create the controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->authorizeResource(LegalProducts::class, 'App\Models\LegalProducts');
    }

    public function index()
    {
        $legalProducts = QueryBuilder::for(LegalProducts::class)
            ->with('court', 'legalProductCategory', 'legalProductType') 
            ->paginate()
            ->appends(request()->query());


        return Inertia::render('Admin/LegalProducts/Index', [
            'legalProducts' => $legalProducts
        ]);
    }

    public function create()
    {
        return Inertia::render('Admin/LegalProducts/Form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|max:255',
            'description' => 'required|max:255',
            'no' => 'required|max:255',
            'year' => 'required|max:4',
            'm_legal_products_types_id' => 'required',
            'm_legal_products_categories_id' => 'required',
            'date_set' => 'required|date|before:tomorrow',
            'invited_date' => 'required|date|before:tomorrow',
            'no_ln' => 'required|max:255',
            'no_tln' => 'required|max:255',
            'status' => 'required',
            'file' => 'required|mimes:pdf|max:5048',
            't_court_id' => 'required',
            'publish_home' => 'required',
        ],[
            'name.required' => 'Nama Dokumen Wajib Diisi',
            'name.max' => 'Maksimal Karakter 255',
            'description.required' => 'Deskripsi Wajib Diisi',
            'description.max' => 'Maksimal Karakter 255',
            'no.required' => 'Nomor Wajib Diisi',
            'year.required' => 'Tahun Wajib Diisi',
            'year.max' => 'Maksimal Karakter 4',
            'm_legal_products_types_id.required' => 'Jenis Dokumen Wajib Diisi',
            'm_legal_products_categories_id.required' => 'Kategori Dokumen Wajib Diisi',
            'date_set.required' => 'Tanggal Ditetapkan Wajib Diisi',
            'date_set.before' => 'Tanggal Yang Dimasukkan Melebihi Waktu Saat Ini',
            'invited_date.required' => 'Tanggal Diundangkan Wajib Diisi',
            'invited_date.before' => 'Tanggal Yang Dimasukkan Melebihi Waktu Saat Ini',
            'no_ln.required' => 'Nomor LN Wajib Diisi',
            'no_ln.max' => 'Maksimal Karakter 255',
            'no_tln.required' => 'Nomor TLN Wajib Diisi',
            'no_tln.max' => 'Maksimal Karakter 255',
            'status.required' => 'Status Wajib Diisi',
            't_court_id.required' => 'Pengadilan Wajib Diisi',
            'publish_home.required' => 'Publikasi Wajib Diisi',
        ]);
        

        DB::beginTransaction();

        try {
            $file_path = '';
            if ($request->file()) {
                $file_name = time() . '_' . $request->file->getClientOriginalName();
                $file_path = $request->file('file')->storeAs('uploads/produk_hukum/' . $request->post('t_court_id'), $file_name, 'public');
            }
            $legalProduct = LegalProducts::create([
                'name' => $request->name,
                'description' => $request->description,
                'file' => $file_path,
                'no' => $request->no,
                'year' => $request->year,
                'no_ln' => $request->no_ln,
                'no_tln' => $request->no_tln,
                'status' => $request->status,
                't_court_id' => $request->t_court_id,
                'law_product_parent' => $request->law_product_parent ?? 0,
                'm_legal_products_types_id' => $request->m_legal_products_types_id,
                'm_legal_products_categories_id' => $request->m_legal_products_categories_id,
                'place_set' => $request->place_set,
                'date_set' => $request->date_set,
                'invited_date' => $request->invited_date,
                'related_regulatory_tags' => json_encode($request->related_regulatory_tags),
                'place_set' => 1,
                'status' => 1,
                'publish_home' => $request->publish_home
            ]);

            $legalProduct->syncTags($request->related_regulatory_tags);

            $content = file_get_contents($request->file('file')->getRealPath());
            $data = base64_encode($content);
            
            $client = ClientBuilder::create()->build();

            $params = [
                'index' => 'produk_hukum',
                'type'  => 'attachment',
                'id'    => $legalProduct->id,
                'pipeline' => 'attachment',  // <----- here
                'body'  => [
                    'data' => $data,
                    'name' => $request->name,
                    'description' => $request->description,
                    'tag' => json_encode($request->related_regulatory_tags, true)
                ]
            ];

            $client->index($params);

            DB::commit();
            
            return Redirect::route('legal-products.index')->with('success', 'Berhasil Simpan Data');

        } catch (\Exception $e) {
            DB::rollBack();
            dd($e);
        }
        
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $legalProduct = LegalProducts::where('id', $id)->with('legalProductType', 'legalProductCategory', 'children')->firstOrFail();
        
        return Inertia::render('Admin/LegalProducts/Show', [
            'legalProduct' => $legalProduct
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $legalProduct = LegalProducts::where('id',$id)->firstOrFail();

        return Inertia::render('Admin/LegalProducts/Form', [
            'legalProduct' => $legalProduct
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|max:255',
            'description' => 'required',
            'no' => 'required|max:255',
            'year' => 'required|max:4',
            'm_legal_products_types_id' => 'required',
            'm_legal_products_categories_id' => 'required',
            'date_set' => 'required|date',
            'invited_date' => 'required|date',
            'no_ln' => 'required|max:255',
            'no_tln' => 'required|max:255',
            'status' => 'required',
            'file' => 'nullable|mimes:pdf|max:5048',
            't_court_id' => 'required',
            'publish' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $file_path = $request->file_update;

            if ($request->file()) {
                $file_name = time() . '_' . $request->file->getClientOriginalName();
                $file_path = $request->file('file')->storeAs('uploads/produk_hukum/' . $request->post('t_court_id'), $file_name, 'public');

                $content = file_get_contents($request->file('file')->getRealPath());
                $data = base64_encode($content);
                
                $client = ClientBuilder::create()->build();

                $params = [
                    'index' => 'produk_hukum',
                    'type'  => 'attachment',
                    'id'    => $id,
                    'pipeline' => 'attachment',  // <----- here
                    'body'  => [
                        'data' => $data,
                        'name' => $request->name,
                        'description' => $request->description,
                        'tag' => json_encode($request->related_regulatory_tags, true)
                    ]
                ];

                $client->update($params);
            } else {
                $client = ClientBuilder::create()->build();

                $params = [
                    'index' => 'produk_hukum',
                    'type'  => 'attachment',
                    'id'    => $id,
                    'body'  => [
                        'doc' => [
                            'name' => $request->name,
                            'description' => $request->description,
                            'tag' => json_encode($request->related_regulatory_tags, true)
                        ]
                    ]
                ];

                $client->update($params);
            }
            
            $legalProduct = LegalProducts::where('id', $id)
                ->update([
                    'name' => $request->name,
                    'description' => $request->description,
                    'file' => $file_path,
                    'no' => $request->no,
                    'year' => $request->year,
                    'no_ln' => $request->no_ln,
                    'no_tln' => $request->no_tln,
                    'status' => $request->status,
                    't_court_id' => $request->t_court_id,
                    'law_product_parent' => $request->law_product_parent ?? 0,
                    'm_legal_products_types_id' => $request->m_legal_products_types_id,
                    'm_legal_products_categories_id' => $request->m_legal_products_categories_id,
                    'place_set' => $request->place_set,
                    'date_set' => $request->date_set,
                    'invited_date' => $request->invited_date,
                    'related_regulatory_tags' => json_encode($request->related_regulatory_tags),
                    'place_set' => 1,
                    'status' => 1
                ]);

            $legalProduct->syncTags($request->related_regulatory_tags);

            DB::commit();
            
            return Redirect::route('legal-products.index')->with('success', 'Berhasil Simpan Data');

        } catch (\Exception $e) {
            DB::rollBack();
            dd($e);
        }
    }

    public function lawProductParentDropdown()
    {
        $datas = LegalProducts::where('law_product_parent', 0)->with('children')->get();

        return $datas;
    }

    public function search(Request $request) {
        $legalProducts = [];
        
        if($request->search) {

            // dd($request);

            $client = ClientBuilder::create()->build();

            $params = [
                'index' => 'produk_hukum',
                'type'  => 'attachment',
                
                'body'  => [
                    'query' => [
                        'bool' => [
                            'should' => [
                                [
                                    'match' => ['attachment.content' => $request->search],
                                ],
                                [
                                    'match' => ['name' => $request->search]
                                ],
                                [
                                    'match' => ['description' => $request->search]
                                ],
                                [
                                    'match' => ['tag' => $request->search]
                                ]
                            ]
                        ],
                        
                    ],
                    'highlight' => [
                        'pre_tags' => ["<em style='background-color:#FF0;'>"], 
                        'post_tags' => ["</em>"],
                        'fields' => [
                            'attachment.content' => new \stdClass(),
                            'name' => new \stdClass(),
                            'description' => new \stdClass(),
                            'tag' => new \stdClass()
                            
                        ],
                        'require_field_match' => false
                        
                    ]
                ]
            ];

            $response = $client->search($params);
            $results   = $response['hits']['hits'];

            $getTags = Tag::containing($request->search)->get();

            foreach($getTags as $item) {
                DB::table('tags')->where("name","ilike","%".$item->name."%")->update(['searched' => ($item->searched + 1) ]);
            }
            
            
            foreach ($results as $key => $value) {
                $data = LegalProducts::where('id', $value['_id'])
                    ->with('legalProductCategory')
                    ->when($request->status, function ($query) use ($request) {
                        return $query->whereIn('status', $request->status);
                    })
                    ->when($request->year, function ($query) use ($request) {
                        return $query->whereIn('year', $request->year);
                    })
                    ->when($request->productType, function ($query) use ($request) {
                        return $query->whereIn('m_legal_products_types_id', $request->productType);
                    })->first();
                
                if($data) {
                    $legalProducts[$key]['name'] = $data->name;
                    $legalProducts[$key]['highlight'] = $value['highlight']['attachment.content'][0] ?? "<em style='background-color:#FF0;'>".$data->description."</em>";
                    $legalProducts[$key]['slug'] = $data->slug;
                    $legalProducts[$key]['category'] = $data->legalProductCategory->name;
                    $legalProducts[$key]['viewed'] = $data->viewed;
                    $legalProducts[$key]['download'] = $data->download;
                }
                
            }
        }

        $legalProductType = LegalProductsTypes::get();
        $legalProductStatus = LegalProductsStatus::where('publish', 1)->get();
        $legalProductYear = LegalProducts::select(DB::raw('year as tahun, count(*) as total_data'))->whereRaw('year::int > extract(year from now())-5')->groupBy('year')->orderBy('year', 'desc')->get();
        

        return Inertia::render('Search_Result', [
            'legalProducts' => $legalProducts,
            'searchParam' => $request->search,
            'legalProductType' => $legalProductType,
            'legalProductStatus' => $legalProductStatus,
            'legalProductYear' => $legalProductYear
        ]);
    }

    public function detail($slug)
    {

        $legalProducts = LegalProducts::where('slug', $slug)->with('legalProductType', 'legalProductCategory', 'children')->firstOrFail();
        
        $legalProductView = LegalProducts::where('id', $legalProducts->id)->first();

        $legalProductView->update([
            'viewed' => $legalProductView->viewed + 1,    
        ]);

        $legalProductStatus = LegalProductsStatus::where('publish', 1)->get();
        $legalProductType = LegalProductsCategories::get();
        return Inertia::render('Legal_Detail_new', [
            'legalProducts' => $legalProducts,
            'searchParam' => '',
            'legalProductType' => $legalProductType,
            'legalProductStatus' => $legalProductStatus
        ]);
    }

    public function downloadFile($slug)
    {
        $legalProduct = LegalProducts::where('slug', $slug)->with('legalProductType')->firstOrFail();
        
        $legalProductView = LegalProducts::where('id', $legalProduct->id)->first();
        try {
            $legalProductView->update([
                'downloaded' => $legalProductView->downloaded + 1,    
            ]);
    
            return response()->download(public_path('storage/'. $legalProductView->file));
        } catch (\Exception $e) {
            dd($e);
        }

        
    }

    public function category($slug) {
        $legalProducts = QueryBuilder::for(LegalProducts::class)
            ->join('m_legal_products_categories','t_legal_products.m_legal_products_categories_id','=','m_legal_products_categories.id')
            // ->where('t_legal_products.m_legal_products_categories_id','=',$id)
            ->where('m_legal_products_categories.slug','=',$slug)
            ->orderBy('t_legal_products.id','DESC')
            ->select('t_legal_products.name as name','t_legal_products.description as highlight','t_legal_products.slug as slug','m_legal_products_categories.name as category','t_legal_products.viewed as viewed','t_legal_products.downloaded as download','m_legal_products_categories.slug as slug_categories')
            ->get();
            
            $legalProductStatus = LegalProductsStatus::where('publish', 1)->get();
            $legalProductType = LegalProductsCategories::get();
            $legalProductYear = LegalProducts::select(DB::raw('year as tahun, count(*) as total_data'))->whereRaw('year::int > extract(year from now())-5')->groupBy('year')->orderBy('year', 'desc')->get();
            return Inertia::render('Search_Result', [
                'legalProducts' => $legalProducts,
                'searchParam' => '',
                'legalProductType' => $legalProductType,
                'legalProductStatus' => $legalProductStatus,
                'legalProductYear' => $legalProductYear
            ]);
    }

    public function categoryType($category, $slug) {
        $legalProducts = QueryBuilder::for(LegalProducts::class)
            ->join('m_legal_products_categories','t_legal_products.m_legal_products_categories_id','=','m_legal_products_categories.id')
            // ->where('t_legal_products.m_legal_products_categories_id','=',$id)
            ->join('m_legal_products_types','t_legal_products.m_legal_products_types_id','=','m_legal_products_types.id')
            ->where('m_legal_products_categories.slug','=',$category)
            ->where('m_legal_products_types.slug','=',$slug)
            ->orderBy('t_legal_products.id','DESC')
            ->select('t_legal_products.name as name','t_legal_products.description as highlight','t_legal_products.slug as slug','m_legal_products_categories.name as category','t_legal_products.viewed as viewed','t_legal_products.downloaded as download','m_legal_products_categories.slug as slug_categories')
            ->get();
        
        $legalProductStatus = LegalProductsStatus::where('publish', 1)->get();
        $legalProductType = LegalProductsCategories::get();
        $legalProductYear = LegalProducts::select(DB::raw('year as tahun, count(*) as total_data'))->whereRaw('year::int > extract(year from now())-5')->groupBy('year')->orderBy('year', 'desc')->get();
            return Inertia::render('Search_Result', [
                'legalProducts' => $legalProducts,
                'searchParam' => '',
                'legalProductType' => $legalProductType,
                'legalProductStatus' => $legalProductStatus,
                'legalProductYear' => $legalProductYear
            ]);
    }

    public function all() {
        $legalProducts = QueryBuilder::for(LegalProducts::class)
            ->join('m_legal_products_categories','t_legal_products.m_legal_products_categories_id','=','m_legal_products_categories.id')
            ->orderBy('t_legal_products.id','DESC')
            ->select('t_legal_products.name as name','t_legal_products.description as highlight','t_legal_products.slug as slug','m_legal_products_categories.name as category','t_legal_products.viewed as viewed','t_legal_products.downloaded as download','m_legal_products_categories.slug as slug_categories', 't_legal_products.updated_at as updated_at')
            ->get();
        $legalProductStatus = LegalProductsStatus::where('publish', 1)->get();
        $legalProductType = LegalProductsCategories::get();
        $legalProductYear = LegalProducts::select(DB::raw('year as tahun, count(*) as total_data'))->whereRaw('year::int > extract(year from now())-5')->groupBy('year')->orderBy('year', 'desc')->get();
        return Inertia::render('Search_Result', [
            'legalProducts' => $legalProducts,
            'searchParam' => '',
            'legalProductType' => $legalProductType,
            'legalProductStatus' => $legalProductStatus,
            'legalProductYear' => $legalProductYear
        ]);
    }
}